<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Competition;
use App\Models\Competitor;
use Illuminate\Support\Facades\Mail;
use App\Mail\CompetitionEnded;
use App\Mail\CompetitorRejected; // تأكد من إنشاء هذا Mailable
use Carbon\Carbon;

class ProcessCompetitionEnds extends Command
{
    /**
     * The name and signature of the console command.
     */
    protected $signature = 'process:competition-ends';

    /**
     * The console command description.
     */
    protected $description = 'Process competitions that have ended, determine winners, and send emails';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Processing ended competitions...');

        // الحصول على المسابقات التي انتهت ولكن لم تتم معالجتها بعد
        $endedCompetitions = Competition::where('is_available', true)
                                ->where('competition_end_date', '<=', Carbon::now())
                                ->where('processed', false)
                                ->get();

        foreach ($endedCompetitions as $competition) {
            // تحديد الفائز (المنافس ذو أعلى رصيد)
            $winner = Competitor::where('competition_id', $competition->id)
                        ->where('status', Competitor::STATUS_IN_COMPETITION)
                        ->orderBy('account_balance', 'desc')
                        ->first();

            if ($winner) {
                // تحديث حالة الفائز إلى 'rejected' وكتابة سبب الرفض
                $winner->update([
                    'status' => Competitor::STATUS_REJECTED,
                    'rejection_reason' => 'الفائز',
                ]);

                // إرسال بريد إلكتروني للفائز
                try {
                    Mail::to($winner->email)->send(new CompetitionEnded($winner, $competition));
                    $this->info("Sent winner email to {$winner->email}");
                } catch (\Exception $e) {
                    $this->error("Failed to send email to {$winner->email}: {$e->getMessage()}");
                }
            }

            // إخراج باقي المنافسين الذين هم في المسابقة
            $nonWinners = Competitor::where('competition_id', $competition->id)
                                ->where('status', Competitor::STATUS_IN_COMPETITION)
                                ->when($winner, function ($query, $winner) {
                                    return $query->where('id', '!=', $winner->id);
                                })
                                ->get();

            foreach ($nonWinners as $competitor) {
                // تحديث حالة المنافس إلى 'rejected' وكتابة سبب الرفض
                $competitor->update([
                    'status' => Competitor::STATUS_REJECTED,
                    'rejection_reason' => 'تم انهاء المسابقة تلقائيا',
                ]);

                // إرسال بريد إلكتروني لإخراجه من المسابقة (اختياري)
                try {
                    Mail::to($competitor->email)->send(new CompetitorRejected($competitor));
                    $this->info("Sent rejection email to {$competitor->email}");
                } catch (\Exception $e) {
                    $this->error("Failed to send email to {$competitor->email}: {$e->getMessage()}");
                }
            }

            // تعيين المسابقة كمعالجة لتجنب إعادة المعالجة
            $competition->update([
                'is_available' => false,
                'processed' => true,
            ]);
        }

        $this->info('Competition ends processing completed.');
    }
}
