<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Competition;
use App\Models\Competitor;
use Carbon\Carbon;

class CompetitionController extends Controller
{
    /**
     * عرض صفحة إدارة المسابقات.
     * يقسم المسابقات إلى مسابقات حالية ومنتهية.
     */
    public function index()
{
    // جلب المسابقات الحالية: is_available = 1
    $currentCompetitions = Competition::where('is_available', true)
        ->orderBy('competition_end_date', 'desc')
        ->get();

    // جلب المسابقات المنتهية: is_available = 0
    $endedCompetitions = Competition::where('is_available', false)
        ->orderBy('competition_end_date', 'desc')
        ->get();

    return view('admin.competitions.index', compact('currentCompetitions', 'endedCompetitions'));
}

    /**
     * عرض نموذج إنشاء مسابقة جديدة.
     */
    public function create()
    {
        return view('admin.competitions.create');
    }

    /**
     * تخزين مسابقة جديدة في قاعدة البيانات.
     */
    public function store(Request $request)
{
    $request->validate([
        'competition_name'      => 'required|string|max:255',
        'is_available'          => 'required|boolean',
        'balance'               => 'required|numeric|min:0',
        'gift'                  => 'required|string|max:255',
        'submission_end_date'   => 'required|date|after:now',
        'competition_end_date'  => 'required|date|after:submission_end_date',
        'account_type'          => 'required|in:demo,live',
        'trading_group'         => 'required|string|max:255',
        'has_stop'              => 'required|boolean',
        'stop_amount'           => 'required_if:has_stop,1|nullable|numeric|min:0',
    ], [
        'competition_name.required'        => 'اسم المسابقة مطلوب.',
        'is_available.required'            => 'حقل "هل المسابقة متاحة" مطلوب.',
        'balance.required'                 => 'حقل "الرصيد البدائي للعملاء" مطلوب.',
        'gift.required'                    => 'حقل "الجائزة" مطلوب.',
        'submission_end_date.required'     => 'تاريخ انتهاء التقديم مطلوب.',
        'submission_end_date.after'        => 'يجب أن يكون تاريخ انتهاء التقديم بعد التاريخ الحالي.',
        'competition_end_date.required'    => 'تاريخ انتهاء المسابقة مطلوب.',
        'competition_end_date.after'       => 'يجب أن يكون تاريخ انتهاء المسابقة بعد تاريخ انتهاء التقديم.',
        'account_type.required'            => 'نوع الحساب مطلوب.',
        'account_type.in'                  => 'نوع الحساب غير صالح.',
        'trading_group.required'           => 'كروب التداول مطلوب.',
        'has_stop.required'                => 'حقل "هل يوجد ستوب؟" مطلوب.',
        'stop_amount.required_if'          => 'مقدار الستور أو الخسارة مطلوب عندما يكون هناك ستوب.',
        'stop_amount.numeric'              => 'مقدار الستور أو الخسارة يجب أن يكون رقمًا.',
        'stop_amount.min'                   => 'مقدار الستور أو الخسارة يجب أن يكون على الأقل 0.',
    ]);

    Competition::create([
        'competition_name'      => $request->competition_name,
        'is_available'          => $request->is_available,
        'balance'               => $request->balance,
        'gift'                  => $request->gift,
        'submission_end_date'   => $request->submission_end_date,
        'competition_end_date'  => $request->competition_end_date,
        'account_type'          => $request->account_type,
        'trading_group'         => $request->trading_group,
        'has_stop'              => $request->has_stop,
        'stop_amount'           => $request->has_stop ? $request->stop_amount : null,
    ]);

    return redirect()->route('competitions.index')->with('success', 'تم إنشاء المسابقة بنجاح.');
}



    /**
     * عرض نموذج تعديل مسابقة موجودة.
     */
    public function edit(Competition $competition)
    {
        return view('admin.competitions.edit', compact('competition'));
    }

    /**
     * تحديث مسابقة موجودة في قاعدة البيانات.
     */
    public function update(Request $request, Competition $competition)
{
    $request->validate([
        'competition_name'      => 'required|string|max:255',
        'is_available'          => 'required|boolean',
        'balance'               => 'required|numeric|min:0',
        'gift'                  => 'required|string|max:255',
        'submission_end_date'   => 'required|date',
        'competition_end_date'  => 'required|date|after_or_equal:submission_end_date',
        'account_type'          => 'required|in:demo,live',
        'trading_group'         => 'required|string|max:255',
        'has_stop'              => 'required|boolean',
        'stop_amount'           => 'required_if:has_stop,1|nullable|numeric|min:0',
    ], [
        'competition_name.required'        => 'اسم المسابقة مطلوب.',
        'is_available.required'            => 'حقل "هل المسابقة متاحة" مطلوب.',
        'balance.required'                 => 'حقل "الرصيد البدائي للعملاء" مطلوب.',
        'gift.required'                    => 'حقل "الجائزة" مطلوب.',
        'submission_end_date.required'     => 'تاريخ انتهاء التقديم مطلوب.',
        'competition_end_date.required'    => 'تاريخ انتهاء المسابقة مطلوب.',
        'competition_end_date.after_or_equal' => 'يجب أن يكون تاريخ انتهاء المسابقة بعد أو يساوي تاريخ انتهاء التقديم.',
        'account_type.required'            => 'نوع الحساب مطلوب.',
        'account_type.in'                  => 'نوع الحساب غير صالح.',
        'trading_group.required'           => 'كروب التداول مطلوب.',
        'has_stop.required'                => 'حقل "هل يوجد ستوب؟" مطلوب.',
        'stop_amount.required_if'          => 'مقدار الستور أو الخسارة مطلوب عندما يكون هناك ستوب.',
        'stop_amount.numeric'              => 'مقدار الستور أو الخسارة يجب أن يكون رقمًا.',
        'stop_amount.min'                   => 'مقدار الستور أو الخسارة يجب أن يكون على الأقل 0.',
    ]);

    $competition->update([
        'competition_name'      => $request->competition_name,
        'is_available'          => $request->is_available,
        'balance'               => $request->balance,
        'gift'                  => $request->gift,
        'submission_end_date'   => $request->submission_end_date,
        'competition_end_date'  => $request->competition_end_date,
        'account_type'          => $request->account_type,
        'trading_group'         => $request->trading_group,
        'has_stop'              => $request->has_stop,
        'stop_amount'           => $request->has_stop ? $request->stop_amount : null,
    ]);

    return redirect()->route('competitions.index')->with('success', 'تم تحديث المسابقة بنجاح.');
}



    /**
     * إنهاء مسابقة موجودة.
     * عند إنهاء المسابقة، يتم تعيين تاريخ انتهاء المسابقة إلى يوم سابق،
     * وتحويل جميع المتسابقين في المسابقة إلى حالة "مرفوضين" مع سبب الرفض.
     */
    public function destroy(Competition $competition)
    {
        // تعيين is_available إلى 0 لإنهاء المسابقة
    $competition->update([
        'is_available' => 0,
    ]);

        // تحويل جميع المتسابقين في المسابقة إلى حالة "مرفوضين" مع سبب الرفض
        Competitor::where('competition_id', $competition->id)
            ->where('status', Competitor::STATUS_IN_COMPETITION)
            ->update([
                'status'            => Competitor::STATUS_REJECTED,
                'rejection_reason' => 'تم إنهاء المسابقة.',
            ]);

        return redirect()->route('competitions.index')->with('success', 'تم إنهاء المسابقة وتحديث حالة المتسابقين.');
    }
}
