<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Competitor;
use App\Models\Competition;
use Illuminate\Validation\Rule;
use Carbon\Carbon;

class SignupController extends Controller
{
    /**
     * عرض نموذج التقديم.
     */
    public function create()
    {
        // جلب المسابقات التي لم تنته فترة التقديم لها بعد
        $competitions = Competition::where('is_available', true)
            ->where('submission_end_date', '>=', Carbon::now())
            ->get();

        return view('signup.create', compact('competitions'));
    }

    /**
     * تخزين المتسابق الجديد.
     */
    public function store(Request $request)
    {
        $request->validate([
            'first_name'       => 'required|string|max:20',
            'last_name'        => 'required|string|max:50',
            'email'            => [
                'required',
                'email',
                Rule::unique('competitors')->where(function ($query) use ($request) {
                    return $query->where('competition_id', $request->competition_id);
                }),
            ],
            'phone'            => 'required|string|max:20',
            'strategy'         => 'required|string', // تم تحويله إلى textarea
            'competition_id'   => 'required|exists:competitions,id',
            'photo'            => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
        ], [
            'email.unique' => 'أنت متقدم بالفعل لهذه المسابقة.',
        ]);

        // التحقق من أن المسابقة لا تزال مفتوحة للتقديم
        $competition = Competition::findOrFail($request->competition_id);
        if ($competition->submission_end_date < Carbon::now()) {
            return redirect()->back()->withErrors(['competition_id' => 'فترة التقديم لهذه المسابقة قد انتهت.']);
        }

        // تخزين الصورة
        if ($request->hasFile('photo')) {
            $imageName = time() . '.' . $request->photo->extension();
            $request->photo->move(public_path('images'), $imageName);
        }

        

        // إنشاء متسابق جديد
        Competitor::create([
            'first_name'       => $request->first_name,
            'last_name'        => $request->last_name,
            'phone'            => $request->phone,
            'strategy'         => $request->strategy, // الآن هو textarea
            'email'            => $request->email,
            'photo'            => $imageName,
            'account_balance'  => 0,
            'current_balance'  => 0,
            'account_number'   => null,
            'trading_password' => null,
            'status'           => Competitor::STATUS_PENDING,
            'competition_id'   => $request->competition_id,
        ]);

        return redirect()->route('home')->with('success', 'تم التقديم بنجاح وسيتم مراجعة طلبك قريباً.');
    }


}
